// Test for being for pull down and pull up resistor button settings
//
// Copyright (C) 2012, Michael Meissner (arduino@the-meissners.org)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

// Standard output pin assignments
// Pin 0:	serial receive (usb)
// Pin 1:	serial transmit (usb)
// Pin 2:	Interrupt 2 via attachInterrupt
// Pin 3:	Interrupt 3 via attachInterrupt, 8-bit PWM output via analogWrite
// Pin 4:	No special use
// Pin 5:	8-bit PWM output via analogWrite
// Pin 6:	8-bit PWM output via analogWrite
// Pin 7:	No special use
// Pin 8:	No special use
// Pin 9:	8-bit PWM output via analogWrite
// Pin 10:	8-bit PWM output via analogWrite, SPI SS
// Pin 11:	8-bit PWM output via analogWrite, SPI MOSI
// Pin 12:	SPI MISO
// Pin 13:	On board LED, SPI SCK

// Note, the Servo library disables analogWrite (PWM) on pins 9 & 10
// whether or not there is a server on those pins

// Standard input pin assignments
// Pin A0:	No special use
// Pin A1:	No special use
// Pin A2:	No special use
// Pin A3:	No special use
// Pin A4:	TWI SDA pin (Wire library)
// Pin A5:	TWI SCL pin (Wire library)

// Debugging
const bool do_print	= true;		// whether to print a summary
const bool init_lights	= true;		// whether to flash the lights at start

// Turn on MrmButton debugging
#define MrmButton_debug do_print

// Meissner button library
#include "MrmButton.h"

// Pin assignments
const int pin_up	= 2;		// on/off switch for pull up resistor
const int pin_down	= 3;		// on/off switch for pull down resistor
const int pin_blink	= 99;		// phony pin for blink
const int pin_pot	= A1;		// 10k Potentiometer
const int pin_led_focus	= 12;		// Led for focus
const int pin_led_fire	= 13;		// Led for fire

// Button hold time
unsigned long last_millis;

// Default timeout in milliseconds if we have no potentiometer
#ifndef POTENTIOMETER_DEFAULT
#define POTENTIOMETER_DEFAULT	250
#endif

//#define HAVE_PULL_DOWN	1
#define HAVE_PULL_UP		1
#define HAVE_POTENTIOMETER	1

#if !defined(HAVE_PULL_DOWN) && !defined(HAVE_PULL_UP)
#define HAVE_BLINK		1
#endif

// Buttons
#ifdef HAVE_PULL_DOWN
MrmButton_pulldown	button_pull_down (pin_down);
#else
MrmButton_nop		button_pull_down (pin_down, LOW);
#endif

#ifdef HAVE_PULL_UP
MrmButton_pullup	button_pull_up (pin_up);
#else
MrmButton_nop		button_pull_up (pin_up, LOW);
#endif

#ifdef HAVE_BLINK
MrmButton_blink		button_blink (pin_blink);
#else
MrmButton_nop		button_blink (pin_blink, LOW);
#endif

#ifdef HAVE_POTENTIOMETER
MrmButton_analog	button_pot (pin_pot);
#else
MrmButton_nop		button_pot (pin_pot, POTENTIOMETER_DEFAULT);
#endif

void
setup (void)
{
  int pot;

  if (do_print)
    {
      Serial.begin (9600);
      Serial.println ("Start");
      Serial.flush ();
    }

  button_pot.setup ();
  button_pull_down.setup ();
  button_pull_up.setup ();
  button_blink.setup ();

  if (do_print)
    Serial.flush ();

  pinMode (pin_led_focus, OUTPUT);
  pinMode (pin_led_fire,  OUTPUT);

  // flash the lights on at start to make sure both are connected
  if (init_lights)
    {
      digitalWrite (pin_led_focus, HIGH);
      digitalWrite (pin_led_fire,  HIGH);

      pot = button_pot.read_value ();

      if (do_print)
	{
	  Serial.print ("Sleep, sleep time = ");
	  Serial.println (pot);
	  Serial.flush ();
	}

      delay (pot);

      if (do_print)
	{
	  Serial.println ("Awake");
	  Serial.flush ();
	}
    }

  digitalWrite (pin_led_focus, LOW);
  digitalWrite (pin_led_fire,  LOW);

  if (do_print)
    {
      Serial.println ("Press the button");
      Serial.flush ();
    }

  last_millis = millis ();
}

void
loop (void)
{
  int value		= 0;
  int num		= -1;
  const char *type	= "";

  if (button_pull_down.read (&value, &num, &type)
      || button_pull_up.read (&value, &num, &type)
      || button_blink.read (&value, &num, &type))
    {
      if (value)
	{
	  digitalWrite (pin_led_focus, HIGH);
	  digitalWrite (pin_led_fire,  LOW);

	  if (do_print)
	    {
	      Serial.print ("Button #");
	      Serial.print (num);
	      Serial.print (", type ");
	      Serial.print (type);
	      Serial.println (", focus");
	      Serial.flush ();
	    }

	  last_millis = millis ();
	}
      else
	{
	  int pot = button_pot.read_value ();

	  digitalWrite (pin_led_focus, HIGH);
	  digitalWrite (pin_led_fire,  HIGH);

	  if (do_print)
	    {
	      unsigned long diff_millis = millis () - last_millis;
	      unsigned long seconds = diff_millis / 1000;
	      unsigned int millis_seconds = (unsigned int)(diff_millis % 1000);
	      unsigned int power_10;

	      Serial.print ("Button #");
	      Serial.print (num);
	      Serial.print (", type ");
	      Serial.print (type);
	      Serial.print (", fire, sleep time = ");
	      Serial.print (pot);
	      Serial.print (", held for ");
	      Serial.print (seconds);
	      Serial.print (".");
	      power_10 = 100;
	      do
		{
		  Serial.print (millis_seconds / power_10);
		  millis_seconds %= power_10;
		  power_10 /= 10;
		}
	      while (power_10 > 0);
	      Serial.println (" seconds");
	      Serial.flush ();
	    }

	  delay (pot);

	  if (do_print)
	    {
	      Serial.print ("Button #");
	      Serial.print (num);
	      Serial.print (", type ");
	      Serial.print (type);
	      Serial.println (", done");
	      Serial.println ("");
	      Serial.flush ();
	    }

	  digitalWrite (pin_led_focus, LOW);
	  digitalWrite (pin_led_fire,  LOW);
	}
    }
}


// HISTORY
// $Log: MrmButtonTest.pde,v $
// Revision 1.1  2012-08-07 13:30:39  michaelmeissner
// Initial version.
//
