// -*- arduino -*-
// Button abstraction library
//
// Copyright (C) 2012-2013, Michael Meissner (arduino@the-meissners.org)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

// This library provides 5 classes:
// MrmButton_pullup	-- Digital button, no resistor, uses internal pullups
// MrmButton_pulldown	-- Digital button, with resistor to ground
// MrmButton_analog	-- Analog button
// MrmButton_blink	-- Dummy button that blinks on and off
// MrmButton_nop	-- Dummy button that returns a constant value

#ifndef MrmButton_h
#define MrmButton_h

#include <stddef.h>
#include <stdlib.h>
#include <inttypes.h>

#if defined(ARDUINO) && ARDUINO >= 100
#include "Arduino.h"

#else
#include "WProgram.h"
#include "WConstants.h"
#endif


// Basic button class
class MrmButton {
protected:
  int pin;			// pin number
  int value;			// key value (LOW/HIGH for digital pins or analog value)
  bool changed;			// whether the last read changed state
  const char *type;		// button type

  // Constructor, only callable by superclasses
  MrmButton (int pin2, const char *type2)
  {
    pin = pin2;
    type = type2;
    value = 0;
    changed = false;
    return;
  }

  // Destructor
  ~ MrmButton (void)
  {
    return;
  }

  // Do all of the processing to read the pin
  virtual void read_internal (void)
  {
#ifdef MrmButton_debug
    if (MrmButton_debug)
      Serial.println ("Virtual read_internal called");
#endif

    value = 0;
    changed = false;
    return;
  }

  // Do any necessary setup for the button
  virtual void setup2 (void)
  {
#ifdef MrmButton_debug
    if (MrmButton_debug)
      Serial.println ("Virtual setup function called");
#endif
  }

public:

  // Button setup
  void setup (void)
  {
#ifdef MrmButton_debug
    if (MrmButton_debug)
      {
	Serial.print ("Setup, pin ");
	Serial.print (pin);
	Serial.print (", type ");
	Serial.println (type);
      }
#endif

    setup2 ();
  }

  // Do a read and return the key value (LOW/HIGH for digital or analog value)
  int read_value (void)
  {
    read_internal ();
    return value;
  }

  // Return whether the state was changed in the last read
  bool state_changed (void)
  {
    return changed;
  }

  // Return the pin number
  int pin_number (void)
  {
    return pin;
  }

  // Return what type of button this is
  const char *
  button_type (void)
  {
    return type;
  }

  // Read the pin
  // Return true if the key state changed
  // Optionally return the key value
  // Optionally return the pin number
  // Optionally return a string giving the button type
  bool read (int *ptr_value		= (int *)0,
	     int *ptr_pin		= (int *)0,
	     const char **ptr_type	= (const char **)0)
  {
    read_internal ();

    if (ptr_value)
      *ptr_value = value;

    if (ptr_pin)
      *ptr_pin = pin;

    if (ptr_type)
      *ptr_type = type;

    return changed;
  }
};


// Button class for buttons with a resistor between ground and the button (pull down)
// digitalRead returns HIGH when the button is pressed

class MrmButton_pulldown : public MrmButton {
protected:

  // Do all of the processing to read the pin
  virtual void read_internal (void)
  {
    int prev_value = value;
    value = digitalRead (pin);
    changed = (value != prev_value);
    return;
  }

public:

  // Constructor
  MrmButton_pulldown (int pin2) : MrmButton (pin2, "pull down")
  {
    value = digitalRead (pin);
    changed = false;
    return;
  }

  // Destructor
  ~ MrmButton_pulldown (void)
  {
    return;
  }

  // Do any necessary setup for the button
  // Register the pin as a digital input pin
  virtual void setup2 (void)
  {
    pinMode (pin, INPUT);
    value = digitalRead (pin);
    return;
  }
};


// Button class for buttons where we tell the Arduino to use the internal pull
// up state and no resistor is needed.
// digitalRead returns LOW when the button is pressed and HIGH otherwise

class MrmButton_pullup : public MrmButton {
protected:

  // Do all of the processing to read the pin
  virtual void read_internal (void)
  {
    int prev_value = value;
    int state = digitalRead (pin);
    value = (state == LOW);
    changed = (value != prev_value);
    return;
  }

public:

  // Constructor
  MrmButton_pullup (int pin2) : MrmButton (pin2, "pull up")
  {
    value = -1;
    changed = false;
    return;
  }

  // Destructor
  ~ MrmButton_pullup (void)
  {
    return;
  }

  // Do any necessary setup for the button
  // Register the pin as a digital input pin & enable internal pull up resistor
  virtual void setup2 (void)
  {
    int state;

#if defined(ARDUINO) && ARDUINO >= 100
    pinMode (pin, INPUT_PULLUP);

#else
    // Pull up method for 0.21
    pinMode (pin, INPUT);
    digitalWrite (pin, HIGH);
#endif

    state = digitalRead (pin);
    value = (state == LOW);
    changed = false;
  }
};


// Button class for analog buttons
// analogRead returns 0..1023

class MrmButton_analog : public MrmButton {
protected:

  // Do all of the processing to read the pin
  virtual void read_internal (void)
  {
    int prev_value = value;
    value = analogRead (pin);
    changed = (value != prev_value);
    return;
  }

public:

  // Constructor
  MrmButton_analog (int pin2) : MrmButton (pin2, "analog")
  {
    value = -1;
    changed = false;
    return;
  }

  // Destructor
  ~ MrmButton_analog (void)
  {
    return;
  }

  // Do any necessary setup for the button
  // Register the pin as a digital input pin & enable internal pull up resistor
  virtual void setup2 (void)
  {
    pinMode (pin, INPUT);
    value = analogRead (pin);
    changed = false;
  }
};


// Button class that is a NOP, and returns a constant value

class MrmButton_nop : public MrmButton {
private:
  int return_value;

protected:

  // Do all of the processing to read the pin
  virtual void read_internal (void)
  {
    value   = return_value;
    changed = false;
    return;
  }

public:

  // Constructor
  MrmButton_nop (int pin2, int return_value2 = LOW) : MrmButton (pin2, "nop")
  {
    changed = false;
    value = return_value = return_value2;
    return;
  }

  // Destructor
  ~ MrmButton_nop (void)
  {
    return;
  }

  // Do any necessary setup for the button
  virtual void setup2 (void)
  {
  }
};


// Button class that periodically turns on and off

#ifndef BLINK_MILLIS_ON
#define BLINK_MILLIS_ON		5000		// on for 5 seconds
#endif

#ifndef BLINK_MILLIS_OFF
#define BLINK_MILLIS_OFF	10000		// off for 10 second
#endif

#ifndef BLINK_INITIAL
#define BLINK_INITIAL		LOW		// inital state
#endif

class MrmButton_blink : public MrmButton {
private:
  unsigned long millis_on;
  unsigned long millis_off;
  unsigned long millis_flip;
  unsigned long millis_now;

protected:

  // Do all of the processing to read the pin
  virtual void read_internal (void)
  {
    unsigned long prev = millis_now;
    unsigned long now = millis ();

    millis_now = now;

    // If it is time to flip or the clock rolled over, flip the button
    if ((now >= millis_flip) || (now < prev))
      {
	changed = true;
	if (value == HIGH)
	  {
	    value = LOW;
	    millis_flip = now + millis_off;
	  }
	else
	  {
	    value = HIGH;
	    millis_flip = now + millis_on;
	  }
      }
    else
      changed = false;

    return;
  }

public:

  // Constructor
  MrmButton_blink (int pin2,
		   int initial		     = BLINK_INITIAL,
		   unsigned long millis_on2  = BLINK_MILLIS_ON,
		   unsigned long millis_off2 = BLINK_MILLIS_OFF) : MrmButton (pin2, "blink")
  {
    pin = pin2;
    changed = false;
    value = initial;
    millis_on = millis_on2;
    millis_off = millis_off2;
    return;
  }

  // Destructor
  ~ MrmButton_blink (void)
  {
    return;
  }

  // Do any necessary setup for the button
  virtual void setup2 (void)
  {
    changed = false;
    millis_now = millis ();
    millis_flip = millis_now + ((value) ? millis_on : millis_off);
  }
};

#endif		/* MrmButton_h */


// HISTORY
// $Log: MrmButton.h,v $
// Revision 1.5  2013-05-23 12:16:56  michaelmeissner
// Update copyright.
//
// Revision 1.4  2012-08-07 11:12:07  michaelmeissner
// Add copyleft
//
// Revision 1.3  2012-07-26 11:58:30  michaelmeissner
// Fix typo
//
// Revision 1.2  2012-07-26 05:44:50  michaelmeissner
// Add #ifdefs for Arduino 1.0
//
// Revision 1.1  2012-07-25 13:23:56  michaelmeissner
// Initial version.
//
