// Telegraph shutter release by Michael Meissner
//
// Copyright (C) 2012-2013, Michael Meissner (arduino@the-meissners.org)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

// Over the last several years, I have disguised my camera to make it look
// like a press camera from the 1930's for various steampunk events.
// After awhile I started adding bits and pieces to the camera, trying to
// various things you find in a smart phone to the setup.  One of the things I
// added was a telegraph key for communication.  A number of people asked me
// whether the telegraph key could fire the camera, and eventually I did.
//
// Originally, I just connected the key into the camera's wired shutter
// release, with one connection for the ground, and the other connection to the
// other two wires (focus and shoot) in a wired shutter release.  The camera
// that I use (Olympus E-5) is a little slow at auto-focusing when I'm using it
// in live view mode.  If I connect both the focus and shoot wires to ground
// together, the E-5 would sometimes shoot before it got a focus lock.
//
// This program runs on the Arduino (UNO R3).  When you make contact on the
// telegraph key, the Arduino through an opto-isolator (4N26JP) connects the
// ground and focus wires which tells the camera to focus.  When you release
// the telegraph key, it keeps the ground and focus wires connected, and
// through another opto-isolator, it connects the ground and shoot wires for a
// period of time.  The period of time is specified by a potentiometer.

// After the shot is done, I use a buzzer to send out the word "fire" in
// Morse code.  The buzzer is a little annoying, and eventually I may decide
// to remove it, or just use LED flashes.  I use the Morse code library written
// by Erik Linder SM0RVV and Mark VandeWettering K6HX.
// http://code.google.com/p/arduino-morse/

// This code also requires the use of my button class library
// http://www.the-meissners.org/arduino/MrmButton.zip

// The current released version of the source should be at
// http://www.the-meissners.org/arduino/ShutterRelease.zip

// Pictures of the Arduino setup for the telegraph key are at:
// http://www.the-meissners.org/2012-small-albums/2012-arduino/small/2012-08-05-23-36-010-arduino.jpg
// http://www.the-meissners.org/2012-small-albums/2012-arduino/small/2012-08-05-23-36-011-arduino.jpg
// http://www.the-meissners.org/2012-small-albums/2012-arduino/small/2012-08-06-23-58-012-arduino.jpg

// Pictures of the evolution of my steampunk cameras can be found at:
// http://www.steampunkmike.org
// http://www.the-meissners.org/2012-small-albums/2012-hide-camera/index.html
// http://www.the-meissners.org/2011-small-albums/2011-hide-camera/index.html
// http://www.the-meissners.org/2010-small-albums/2010-pimp-my-ep2/index.html#hidecamera

// Standard output pin assignments
// Pin 0:	serial receive (usb)
// Pin 1:	serial transmit (usb)
// Pin 2:	Interrupt 2 via attachInterrupt
// Pin 3:	Interrupt 3 via attachInterrupt, 8-bit PWM output via analogWrite
// Pin 4:	No special use
// Pin 5:	8-bit PWM output via analogWrite
// Pin 6:	8-bit PWM output via analogWrite
// Pin 7:	No special use
// Pin 8:	No special use
// Pin 9:	8-bit PWM output via analogWrite
// Pin 10:	8-bit PWM output via analogWrite, SPI SS
// Pin 11:	8-bit PWM output via analogWrite, SPI MOSI
// Pin 12:	SPI MISO
// Pin 13:	On board LED, SPI SCK

// Note, the Servo library disables analogWrite (PWM) on pins 9 & 10
// whether or not there is a server on those pins

// Standard input pin assignments
// Pin A0:	No special use
// Pin A1:	No special use
// Pin A2:	No special use
// Pin A3:	No special use
// Pin A4:	TWI SDA pin (Wire library)
// Pin A5:	TWI SCL pin (Wire library)

// Target machine conditionals:
#ifdef __MK20DX128__
#define HAVE_TEENSY_3_0				// run on Paul Stoffregen's ARM Cortex M4 based teensy 3.0

#else
#define HAVE_ARDUINO_UNO_R3			// run on Arduino UNO r3
#endif

// Conditional compilation controls
#define HAVE_MORSE 		1		// enable morse code
#define HAVE_MORSE_LIGHT	0		// use light instead of buzzer
#define HAVE_POTENTIOMETER	1		// enable using potentiometer

// Debugging
const bool do_print		= true;		// whether to print a summary
const bool init_lights		= true;		// whether to flash the lights at start
const int  init_delay		= 1000;		// initial delay time

// Turn on MrmButton debugging
#define MrmButton_debug do_print

// Meissner button library
#include "MrmButton.h"

// Pin assignments
const int pin_button		=  2;		// on/off switch on protoboard
const int pin_telegraph		=  3;		// telegraph key
const int pin_buzzer		=  4;		// passive buzzer
const int pin_cam_focus		=  8;		// optocoupler for focus
const int pin_cam_fire		=  9;		// optocoupler for fire
const int pin_dip_morse		= 11;		// dip switch, use morse buzzer/light
const int pin_led_focus		= 12;		// led for focus
const int pin_led_fire		= 13;		// led for fire
const int pin_led_morse		= pin_led_fire;	// led for morse code action
const int pin_pot		= A1;		// 10k Potentiometer

// Button hold time
unsigned long last_millis;

// Default timeout in milliseconds if we have no potentiometer
#ifndef POTENTIOMETER_DEFAULT
#define POTENTIOMETER_DEFAULT	250
#endif

// Button/telegraph key
MrmButton_pullup	button    (pin_button);
MrmButton_pullup	telegraph (pin_telegraph);

#if HAVE_POTENTIOMETER
MrmButton_analog	potentiometer (pin_pot);
#else
MrmButton_nop		potentiometer (pin_pot, POTENTIOMETER_DEFAULT);
#endif

#if HAVE_MORSE
// Arduino Morse Library
// Written by Erik Linder SM0RVV and Mark VandeWettering K6HX
#include "Morse.h"

// Morse speed
const int		morse_speed_light  = 12;
const int		morse_speed_buzzer = 24;

// Morse message
char			morse_msg[]	= "fire";

#if HAVE_MORSE_LIGHT
const char		morse_type[]	= "Morse light: ";
Morse			morse_buzzer_or_light (pin_led_morse, morse_speed_light, 0);

#else
const char		morse_type[]	= "Morse buzzer: ";
Morse			morse_buzzer_or_light (pin_buzzer, morse_speed_buzzer, 1);
#endif

MrmButton_pullup	do_morse_action  (pin_dip_morse);
#endif

// Print milliseconds as seconds.fraction

void
print_milliseconds (unsigned long m)
{
  unsigned long seconds = m / 1000;
  unsigned int millis_seconds = (unsigned int)(m % 1000);
  unsigned int power_10;

  Serial.print (seconds);
  Serial.print (".");
  power_10 = 100;
  do
    {
      Serial.print (millis_seconds / power_10);
      millis_seconds %= power_10;
      power_10 /= 10;
    }
  while (power_10 > 0);
}

void
setup (void)
{
  if (do_print)
    {
      Serial.begin (9600);
      Serial.println ("Start");
      Serial.flush ();
    }

  button.setup ();
  telegraph.setup ();
  potentiometer.setup ();

  pinMode (pin_led_focus, OUTPUT);
  pinMode (pin_led_fire,  OUTPUT);

  pinMode (pin_cam_focus, OUTPUT);
  pinMode (pin_cam_fire,  OUTPUT);

  digitalWrite (pin_cam_focus, LOW);
  digitalWrite (pin_cam_fire,  LOW);

#if HAVE_MORSE
  do_morse_action.setup ();
  pinMode (pin_buzzer, OUTPUT);
  digitalWrite (pin_buzzer, LOW);

#if HAVE_MORSE_LIGHT
  if (pin_led_morse != pin_led_fire && pin_led_morse != pin_led_focus)
    {
      pinMode (pin_led_morse,  OUTPUT);
    }
#endif
#endif

  // flash the lights on at start to make sure both are connected
  if (init_lights)
    {
      digitalWrite (pin_led_focus, HIGH);
      digitalWrite (pin_led_fire,  HIGH);

#if HAVE_MORSE && HAVE_MORSE_LIGHT
      if (pin_led_morse != pin_led_fire && pin_led_morse != pin_led_focus)
	{
	  digitalWrite (pin_led_morse, HIGH);
	}
#endif

      if (do_print)
	{
	  Serial.print ("Sleep, sleep time = ");
	  print_milliseconds ((unsigned long)init_delay);
	  Serial.println (" seconds");
	  Serial.flush ();
	}

      delay (init_delay);

      if (do_print)
	{
	  Serial.println ("Awake");
	  Serial.flush ();
	}
    }

  digitalWrite (pin_led_focus, LOW);
  digitalWrite (pin_led_fire,  LOW);

#if HAVE_MORSE && HAVE_MORSE_LIGHT
  if (pin_led_morse != pin_led_fire && pin_led_morse != pin_led_focus)
    {
      digitalWrite (pin_led_morse, LOW);
    }
#endif

  if (do_print)
    {
      Serial.println ("Press the button");
      Serial.flush ();
    }

  last_millis = millis ();
}

void
loop (void)
{
  int value	= 0;
  int num	= -1;

  if (button.read (&value, &num) || telegraph.read (&value, &num))
    {
      if (value)
	{
	  digitalWrite (pin_led_focus, HIGH);
	  digitalWrite (pin_led_fire,  LOW);

	  digitalWrite (pin_cam_focus, HIGH);
	  digitalWrite (pin_cam_fire,  LOW);

	  if (do_print)
	    {
	      Serial.print ("Button #");
	      Serial.print (num);
	      Serial.println (", focus");
	      Serial.flush ();
	    }

	  last_millis = millis ();
	}
      else
	{
	  int pot = potentiometer.read_value ();

	  digitalWrite (pin_led_focus, HIGH);
	  digitalWrite (pin_led_fire,  HIGH);

	  digitalWrite (pin_cam_focus, HIGH);
	  digitalWrite (pin_cam_fire,  HIGH);

	  if (do_print)
	    {
	      unsigned long held_time = millis () - last_millis;

	      Serial.print ("Button #");
	      Serial.print (num);
	      Serial.print (", fire, sleep time = ");
	      print_milliseconds ((unsigned long)pot);
	      Serial.print (" seconds, button held for ");
	      print_milliseconds (held_time);
	      Serial.println (" seconds");
	      Serial.flush ();
	    }

	  delay (pot);

	  if (do_print)
	    {
	      Serial.print ("Button #");
	      Serial.print (num);
	      Serial.println (", done");
	      Serial.println ("");
	      Serial.flush ();
	    }

	  digitalWrite (pin_led_focus, LOW);
	  digitalWrite (pin_led_fire,  LOW);

	  digitalWrite (pin_cam_focus, LOW);
	  digitalWrite (pin_cam_fire,  LOW);

#if HAVE_MORSE
	  if (do_morse_action.read_value ())
	    {
	      if (do_print)
		{
		  Serial.print (morse_type);
		  Serial.println (morse_msg);
		  Serial.flush ();
		}

	      morse_buzzer_or_light.sendmsg (morse_msg);

#if HAVE_MORSE_LIGHT
	      digitalWrite (pin_led_morse, LOW);
#endif
	    }

	  else if (do_print)
	    {
	      Serial.println ("Morse: disabled.");
	    }
#endif
	}
    }
}


// HISTORY
// $Log: ShutterRelease.pde,v $
// Revision 1.9  2013-05-23 12:16:37  michaelmeissner
// Update copyright.
//
// Revision 1.8  2012-10-16 03:24:07  michaelmeissner
// Reenable buzzer, debug print statements.
//
// Revision 1.7  2012-10-15 20:10:28  michaelmeissner
// Add #ifdef for teensy
//
// Revision 1.6  2012-10-13 14:15:00  michaelmeissner
// Add support for a dip switch to enable/disable morse action.
//
// Revision 1.5  2012-08-09 13:22:09  michaelmeissner
// Add #ifdef for using LED instead of buzzer, and make it default.
//
// Revision 1.4  2012-08-07 11:57:26  michaelmeissner
// Add links to where the code and library can be downloaded from
//
// Revision 1.3  2012-08-07 11:48:18  michaelmeissner
// Add copyleft and expanded comments.
//
// Revision 1.2  2012-08-06 01:59:27  michaelmeissner
// Do telegraph buzzer after finishing the delay to fire the camera
//
// Revision 1.1  2012-08-06 01:52:17  michaelmeissner
// Initial version.
//
